<?php
namespace Trs;

use tree_table_rate;
use Trs\Services\ApiService;
use Trs\Services\IisCompatService;
use Trs\Services\ServiceInstaller;
use Trs\Services\StatsService;
use Trs\Services\UpdateService;
use Trs\Services\UpgradeService;
use Trs\Woocommerce\ShippingMethod;
use WC_Cache_Helper;
use WC_Shipping_Zones;


class Loader
{
    public function __construct(PluginMeta $pluginMeta)
    {
        $this->pluginMeta = $pluginMeta;
    }
    
    public function bootstrap()
    {
        $this->installServices();
        add_filter('woocommerce_shipping_methods', array($this, '_registerShippingMethods'));
        add_action('init', array($this, '_init'), PHP_INT_MAX);
    }

    public function _init()
    {
        add_filter("plugin_action_links_{$this->pluginMeta->getPluginBasename()}", array($this, '_injectSettingsLink'));
        $this->fixIncorrectHidingOfShippingSectionWhenNoShippingZoneMethodsDefined();

        // On the plugin settings page only
        if (($method = self::editingShippingMethod()) !== null) {

            $enqueueAssets = new EnqueueAssets($method, $this->pluginMeta);
            add_action('admin_enqueue_scripts', $enqueueAssets, PHP_INT_MAX);

            self::removeConflictingScripts();
        }
    }

    public function _registerShippingMethods($shippingMethods)
    {
        static $shippingMethod;

        if (!isset($shippingMethod)) {
            $shippingMethod = new tree_table_rate();
        }

        $shippingMethods[tree_table_rate::className()] = $shippingMethod;

        return $shippingMethods;
    }

    public function _injectSettingsLink($links)
    {
        $settingsUrl = admin_url('admin.php?page=wc-settings&tab=shipping&section='.rawurlencode(tree_table_rate::className()));
        array_unshift($links, '<a href="'.esc_html($settingsUrl).'">'.__('Settings').'</a>');

        return $links;
    }

    /**
     * @return ShippingMethod|null
     */
    static private function editingShippingMethod()
    {
        if (@$_GET['section'] === tree_table_rate::className()) {
            return new ShippingMethod(0);
        }

        if (($instanceId = @$_REQUEST['instance_id']) !== null &&
            class_exists('\\WC_Shipping_Zones') &&
            ($method = WC_Shipping_Zones::get_shipping_method($instanceId)) &&
            ($method instanceof tree_table_rate)) {

            return $method;
        }

        return null;
    }

    static private function removeConflictingScripts()
    {
        // Compatibility with Virtue theme 3.2.2 (https://wordpress.org/themes/virtue/)
        remove_action('admin_enqueue_scripts', 'kadence_admin_scripts');

        // Compatibility with Woocommerce Product Tab Pro 1.8.0 (http://codecanyon.net/item/woocommerce-tabs-pro-extra-tabs-for-product-page/8218941)
        remove_action('admin_print_footer_scripts', '_hc_tinymce_footer_scripts');
    }


    private $pluginMeta;

    private function fixIncorrectHidingOfShippingSectionWhenNoShippingZoneMethodsDefined()
    {
        $trv = WC_Cache_Helper::get_transient_version('shipping');

        // WC before 3.6.0
        add_filter(
            'transient_wc_shipping_method_count_1_' . $trv,
            function($count) {
                return min(1, $count);
            },
            10, 2
        );

        // WC 3.6.0+
        add_filter(

            'transient_wc_shipping_method_count_legacy',

            function($value) use($trv) {
                static $running = false;

                if ($running) return $value;
                $running = true;

                if (!isset($value['value'], $value['version']) ||
                    $value['value'] == 0 ||
                    $value['version'] !== $trv) {

                    $count = max(1, wc_get_shipping_method_count(true));
                    $value['value'] = $count;
                    $value['version'] = $trv;
                }

                $running = false;

                return $value;
            },
            PHP_INT_MAX
        );
    }

    private function installServices()
    {
        $services = array(
            new IisCompatService($this->pluginMeta),
            new UpdateService($this->pluginMeta),
            new StatsService($this->pluginMeta),
            new UpgradeService($this->pluginMeta),
            new ApiService(),
        );

        $installer = new ServiceInstaller();
        foreach ($services as $service) {
            $installer->installIfReady($service);
        }
    }
}