<?php
namespace Trs\Services;

use BadMethodCallException;
use Trs\Migration\UberMigration;
use Trs\Migration\Storage\WordpressOptionsStorage;
use Trs\PluginMeta;
use Trs\Services\Interfaces\IService;
use Trs\Services\Interfaces\IServiceReady;
use TrsVendors\Dgm\WcTools\WcTools;


class UpgradeService implements IService, IServiceReady
{
    public function __construct(PluginMeta $pluginMeta)
    {
        $this->pluginMeta = $pluginMeta;
    }

    public function install()
    {
        $message = null;
        if (!$this->ready($message)) {
            throw new BadMethodCallException($message);
        }

        if (did_action('plugins_loaded')) {
            $this->maybeUpgrade();
        } else {
            add_action('plugins_loaded', array($this, 'maybeUpgrade'));
        }
    }

    public function ready(&$message = null)
    {
        if (!in_array(
            'woocommerce/woocommerce.php',
            apply_filters('active_plugins', get_option('active_plugins'))
        )) {
            $message = 'Active WooCommerce environment required in order to perform upgrades';
            return false;
        }

        return true;
    }

    public function maybeUpgrade()
    {
        $updateCurrentVersion = false;

        $currentVersion = $this->pluginMeta->getVersion();

        $previousVersion = get_option('trs_version');
        if (empty($previousVersion)) {
            $previousVersion = '1.5.0';
            $updateCurrentVersion = true;
        }

        if (version_compare($previousVersion, $currentVersion, '<')) {

            $upgradeScripts = glob($this->pluginMeta->getMigrationsPath('*.php'), GLOB_NOSORT);
            natsort($upgradeScripts);
            
            $migrations = array();
            foreach ($upgradeScripts as $script) {
                $migrationFromVersion = pathinfo($script, PATHINFO_FILENAME);
                if (version_compare($previousVersion, $migrationFromVersion, '<=')) {
                    /** @noinspection PhpIncludeInspection */
                    $migrations[] = include($script);
                }
            }

            if ($migrations) {
                $migration = new UberMigration($migrations, $previousVersion);
                $wpOptions = new WordpressOptionsStorage($GLOBALS['wpdb']);
                $migration->migrate($wpOptions);
            }
            
            // Although, in theory, we don't need to purge shipping cache since we always expect to produce
            // a similar functioning config after migrations, in practice, we'd better allow a user to test
            // a new config right after migration in case there is any issue with that rather than showing
            // results cached from a previous config.
            WcTools::purgeWoocommerceShippingCache();
        }

        if ($updateCurrentVersion || $previousVersion !== $currentVersion) {
            update_option('trs_version', $currentVersion);
        }
    }


    private $pluginMeta;
}